<?php
session_start();
if (!isset($_SESSION['super_admin_id'])) {
    header("Location: login.php");
    exit();
}

require_once __DIR__ . '/../config/database.php';

$database = new Database();
$db = $database->getConnection();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_plan'])) {
        // Add new plan
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $price_monthly = floatval($_POST['price_monthly']);
        $price_yearly = floatval($_POST['price_yearly']);
        $max_staff = intval($_POST['max_staff']);
        $max_members = intval($_POST['max_members']);
        $max_sms = intval($_POST['max_sms']);
        $is_popular = isset($_POST['is_popular']) ? 1 : 0;

        // Process features
        $features = [];
        foreach ($_POST['features'] as $feature) {
            if (!empty(trim($feature))) {
                $features[] = trim($feature);
            }
        }

        try {
            $stmt = $db->prepare("INSERT INTO subscription_plans (name, description, price_monthly, price_yearly, max_staff, max_members, max_sms, features, is_popular) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $name,
                $description,
                $price_monthly,
                $price_yearly,
                $max_staff,
                $max_members,
                $max_sms,
                json_encode($features),
                $is_popular
            ]);

            $_SESSION['success'] = "Plan added successfully!";
        } catch (Exception $e) {
            $_SESSION['error'] = "Error adding plan: " . $e->getMessage();
        }

    } elseif (isset($_POST['edit_plan'])) {
        // Update plan
        $plan_id = intval($_POST['plan_id']);
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $price_monthly = floatval($_POST['price_monthly']);
        $price_yearly = floatval($_POST['price_yearly']);
        $max_staff = intval($_POST['max_staff']);
        $max_members = intval($_POST['max_members']);
        $max_sms = intval($_POST['max_sms']);
        $is_popular = isset($_POST['is_popular']) ? 1 : 0;

        // Process features
        $features = [];
        foreach ($_POST['features'] as $feature) {
            if (!empty(trim($feature))) {
                $features[] = trim($feature);
            }
        }

        try {
            $stmt = $db->prepare("UPDATE subscription_plans SET name = ?, description = ?, price_monthly = ?, price_yearly = ?, max_staff = ?, max_members = ?, max_sms = ?, features = ?, is_popular = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([
                $name,
                $description,
                $price_monthly,
                $price_yearly,
                $max_staff,
                $max_members,
                $max_sms,
                json_encode($features),
                $is_popular,
                $plan_id
            ]);

            $_SESSION['success'] = "Plan updated successfully!";
        } catch (Exception $e) {
            $_SESSION['error'] = "Error updating plan: " . $e->getMessage();
        }
    }

    header("Location: plans.php");
    exit();
}

// Handle actions via GET
if (isset($_GET['action'])) {
    $action = $_GET['action'];
    $id = intval($_GET['id']);

    switch ($action) {
        case 'delete':
            try {
                $stmt = $db->prepare("DELETE FROM subscription_plans WHERE id = ?");
                $stmt->execute([$id]);
                $_SESSION['success'] = "Plan deleted successfully!";
            } catch (Exception $e) {
                $_SESSION['error'] = "Error deleting plan: " . $e->getMessage();
            }
            break;

        case 'toggle':
            try {
                $stmt = $db->prepare("UPDATE subscription_plans SET status = IF(status='active', 'inactive', 'active') WHERE id = ?");
                $stmt->execute([$id]);
                $_SESSION['success'] = "Plan status updated!";
            } catch (Exception $e) {
                $_SESSION['error'] = "Error updating plan status: " . $e->getMessage();
            }
            break;
    }

    header("Location: plans.php");
    exit();
}

// Get all plans
$stmt = $db->query("SELECT * FROM subscription_plans ORDER BY price_monthly ASC");
$plans = $stmt->fetchAll(PDO::FETCH_ASSOC);

// If no plans exist, insert default ones
if (empty($plans)) {
    $default_plans = [
        [
            'name' => 'Basic',
            'description' => 'Perfect for small gyms just getting started',
            'price_monthly' => 80.00,
            'price_yearly' => 760.00,
            'max_staff' => 4,
            'max_members' => 500,
            'max_sms' => 100,
            'features' => [
                'Staff Members (4)',
                'Appointment Booking (Unlimited)',
                'Client Management (Up to 500 clients)',
                'SMS Notifications (100 per month)',
                'Email Marketing (Basic)',
                'Reports & Analytics (Basic)',
                'Online Booking Widget',
                'Mobile App Access',
                'Custom Branding',
                'Priority Support'
            ],
            'is_popular' => false
        ],
        [
            'name' => 'Standard',
            'description' => 'Ideal for growing gyms with multiple staff',
            'price_monthly' => 120.00,
            'price_yearly' => 1152.00,
            'max_staff' => 8,
            'max_members' => 2000,
            'max_sms' => 500,
            'features' => [
                'Staff Members (8)',
                'Appointment Booking (Unlimited)',
                'Client Management (Up to 2000 clients)',
                'SMS Notifications (500 per month)',
                'Email Marketing (Advanced)',
                'Reports & Analytics (Advanced)',
                'Online Booking Widget',
                'Mobile App Access',
                'Custom Branding',
                'Priority Support'
            ],
            'is_popular' => true
        ],
        [
            'name' => 'Premium',
            'description' => 'Advanced features for established gyms',
            'price_monthly' => 180.00,
            'price_yearly' => 1726.00,
            'max_staff' => 15,
            'max_members' => 0,
            'max_sms' => 2000,
            'features' => [
                'Staff Members (15)',
                'Appointment Booking (Unlimited)',
                'Client Management (Unlimited)',
                'SMS Notifications (2000 per month)',
                'Email Marketing (Premium)',
                'Reports & Analytics (Premium)',
                'Online Booking Widget',
                'Mobile App Access',
                'Custom Branding',
                'Priority Support',
                'API Access',
                'Multi-location Support (Up to 3 locations)'
            ],
            'is_popular' => false
        ]
    ];

    foreach ($default_plans as $plan_data) {
        $stmt = $db->prepare("INSERT INTO subscription_plans (name, description, price_monthly, price_yearly, max_staff, max_members, max_sms, features, is_popular) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $plan_data['name'],
            $plan_data['description'],
            $plan_data['price_monthly'],
            $plan_data['price_yearly'],
            $plan_data['max_staff'],
            $plan_data['max_members'],
            $plan_data['max_sms'],
            json_encode($plan_data['features']),
            $plan_data['is_popular']
        ]);
    }

    // Reload plans
    $stmt = $db->query("SELECT * FROM subscription_plans ORDER BY price_monthly ASC");
    $plans = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get plan data for edit (if requested via GET)
$edit_plan = null;
if (isset($_GET['edit_id'])) {
    $edit_id = intval($_GET['edit_id']);
    $stmt = $db->prepare("SELECT * FROM subscription_plans WHERE id = ?");
    $stmt->execute([$edit_id]);
    $edit_plan = $stmt->fetch(PDO::FETCH_ASSOC);
}
?>

<?php include 'includes/header.php'; ?>
<?php include 'includes/sidebar.php'; ?>
<?php include 'includes/navbar.php'; ?>

<div class="main-panel">
    <div class="content-wrapper">
        <div class="page-header flex-wrap">
            <div class="header-left">
                <h3 class="page-title">Subscription Packages</h3>
                <p class="text-muted">Manage your gym management system pricing plans</p>
            </div>
            <div class="header-right d-flex flex-wrap mt-2 mt-sm-0">
                <button type="button" class="btn btn-primary mt-2 mt-sm-0 pt-3 pb-3 
           d-flex justify-content-center align-items-center" data-bs-toggle="modal" data-bs-target="#addPlanModal">
                    Add New Plan
                </button>

            </div>
        </div>

        <!-- Success/Error Messages -->
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success'];
                unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error'];
                unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <!-- Plans Grid -->
        <div class="row">
            <?php foreach ($plans as $plan):
                $features = json_decode($plan['features'], true);
                $is_popular = $plan['is_popular'];
                $status_class = $plan['status'] == 'active' ? 'badge-success' : 'badge-danger';

                // Define card colors based on plan
                $card_class = '';
                $price_color = '';
                $badge_color = '';

                switch ($plan['name']) {
                    case 'Basic':
                        $card_class = 'basic-plan';
                        $price_color = 'text-primary';
                        $badge_color = 'bg-primary';
                        break;
                    case 'Standard':
                        $card_class = 'standard-plan';
                        $price_color = 'text-success';
                        $badge_color = 'bg-primary';
                        break;
                    case 'Premium':
                        $card_class = 'premium-plan';
                        $price_color = 'text-warning';
                        $badge_color = 'bg-primary';
                        break;
                    default:
                        $card_class = 'custom-plan';
                        $price_color = 'text-info';
                        $badge_color = 'bg-info';
                }
                ?>
                <div class="col-xl-4 col-lg-6 col-md-12 grid-margin stretch-card">
                    <div class="card pricing-card <?php echo $card_class; ?> <?php echo $is_popular ? 'popular' : ''; ?>">
                        <?php if ($is_popular): ?>
                            <div class="popular-badge">Most Popular</div>
                        <?php endif; ?>

                        <div class="card-header <?php echo $badge_color; ?> text-white text-center py-4">
                            <h5 style="font-size: 22px;" class="mb-0"><?php echo $plan['name']; ?></h5>
                            <p class="mb-0 opacity-75"><?php echo $plan['description']; ?></p>
                        </div>

                        <div class="card-body">
                            <div class="price-section text-center mb-4">
                              <div class="price-section text-center mb-4">
    <h2 class="price" style="color:#0034c4; font-size:35px !important;">
       Rs. <?php echo $plan['price_monthly']; ?><small>/month</small>
    </h2>
</div>


                                <p class="text-muted">or <?php echo $plan['price_yearly']; ?>/year <span
                                        class="text-success">(Save 20%)</span></p>
                            </div>

<div class="members-count text-center mb-4">
    <span class="badge <?php echo $badge_color; ?> p-2" 
          style="background-color:#0034c4 !important; color:#ffffff !important;">
        <i class="mdi mdi-account-multiple me-1"></i>
        <?php echo $plan['max_members'] == 0 ? 'Unlimited' : $plan['max_members']; ?> Gym Members
    </span>
</div>

                            <div class="features-list mb-4">
                                <?php foreach ($features as $feature): ?>
                                    <div class="d-flex align-items-center mb-3">
                                        <i class="mdi mdi-check-circle-outline text-success me-3 fs-5"></i>
                                        <span class="feature-text"><?php echo $feature; ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <div class="action-buttons mt-4">
                                <div class="btn-group w-100" role="group">
                                    <a href="plans.php?edit_id=<?php echo $plan['id']; ?>" class="btn btn-outline-primary">
                                        <i class="mdi mdi-pencil"></i> Edit
                                    </a>
                                    <button type="button"
                                        class="btn btn-<?php echo $plan['status'] == 'active' ? 'warning' : 'success'; ?>"
                                        onclick="togglePlan(<?php echo $plan['id']; ?>, '<?php echo $plan['status']; ?>')">
                                        <i
                                            class="mdi mdi-<?php echo $plan['status'] == 'active' ? 'pause' : 'play'; ?>"></i>
                                    </button>
                                    <button type="button" class="btn btn-outline-danger"
                                        onclick="deletePlan(<?php echo $plan['id']; ?>, '<?php echo $plan['name']; ?>')">
                                        <i class="mdi mdi-delete"></i> Delete
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <?php include 'includes/footer.php'; ?>
</div>

<!-- Add Plan Modal -->
<div class="modal fade" id="addPlanModal" tabindex="-1" aria-labelledby="addPlanModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header text-black">
                <h5 class="modal-title" id="addPlanModalLabel">Add New Subscription Plan</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"
                    aria-label="Close"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Plan Name *</label>
                                <input type="text" name="name" class="form-control" required
                                    placeholder="e.g., Basic, Standard, Premium">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Monthly Price (PKR) *</label>
                                <input type="number" name="price_monthly" class="form-control" step="0.01" required
                                    placeholder="80.00">
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Yearly Price (PKR)</label>
                                <input type="number" name="price_yearly" class="form-control" step="0.01"
                                    placeholder="760.00">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Max Staff Members</label>
                                <input type="number" name="max_staff" class="form-control" value="4" placeholder="4">
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Max Members</label>
                                <input type="number" name="max_members" class="form-control" value="500"
                                    placeholder="500">
                                <small class="text-muted">Set 0 for unlimited members</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="form-label">Max SMS per Month</label>
                                <input type="number" name="max_sms" class="form-control" value="100" placeholder="100">
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Description</label>
                        <textarea name="description" class="form-control" rows="2"
                            placeholder="Brief description of the plan"></textarea>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Features</label>
                        <div id="add-features-container">
                            <div class="input-group mb-2">
                                <input type="text" name="features[]" class="form-control"
                                    placeholder="Add a feature (e.g., Staff Members (4))">
                                <button type="button" class="btn btn-success add-feature-btn">
                                    <i class="mdi mdi-plus"></i>
                                </button>
                            </div>
                        </div>
                        <small class="text-muted">Click + to add more features</small>
                    </div>

                    <div class="form-group">
                        <div class="form-check form-switch">
                            <input type="checkbox" name="is_popular" class="form-check-input" id="add_is_popular"
                                role="switch">
                            <label class="form-check-label" for="add_is_popular">
                                Mark as Popular Plan (Will be highlighted)
                            </label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_plan" class="btn btn-primary">
                        <i class="mdi mdi-plus-circle"></i> Create Plan
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Plan Modal (Will auto-show if edit_id is set) -->
<div class="modal fade" id="editPlanModal" tabindex="-1" aria-labelledby="editPlanModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="editPlanModalLabel">Edit Subscription Plan</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"
                    aria-label="Close"></button>
            </div>
            <?php if ($edit_plan): ?>
                <form method="POST">
                    <input type="hidden" name="plan_id" value="<?php echo $edit_plan['id']; ?>">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Plan Name *</label>
                                    <input type="text" name="name" class="form-control" required
                                        value="<?php echo htmlspecialchars($edit_plan['name']); ?>">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Monthly Price (PKR) *</label>
                                    <input type="number" name="price_monthly" class="form-control" step="0.01" required
                                        value="<?php echo $edit_plan['price_monthly']; ?>">
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Yearly Price (PKR)</label>
                                    <input type="number" name="price_yearly" class="form-control" step="0.01"
                                        value="<?php echo $edit_plan['price_yearly']; ?>">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Max Staff Members</label>
                                    <input type="number" name="max_staff" class="form-control"
                                        value="<?php echo $edit_plan['max_staff']; ?>">
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Max Members</label>
                                    <input type="number" name="max_members" class="form-control"
                                        value="<?php echo $edit_plan['max_members']; ?>">
                                    <small class="text-muted">Set 0 for unlimited members</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Max SMS per Month</label>
                                    <input type="number" name="max_sms" class="form-control"
                                        value="<?php echo $edit_plan['max_sms']; ?>">
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Description</label>
                            <textarea name="description" class="form-control"
                                rows="2"><?php echo htmlspecialchars($edit_plan['description']); ?></textarea>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Features</label>
                            <div id="edit-features-container">
                                <?php
                                $features = json_decode($edit_plan['features'], true);
                                foreach ($features as $feature):
                                    ?>
                                    <div class="input-group mb-2">
                                        <input type="text" name="features[]" class="form-control"
                                            value="<?php echo htmlspecialchars($feature); ?>">
                                        <button type="button" class="btn btn-danger remove-feature-btn">
                                            <i class="mdi mdi-minus"></i>
                                        </button>
                                    </div>
                                <?php endforeach; ?>
                                <div class="input-group mb-2">
                                    <input type="text" name="features[]" class="form-control" placeholder="Add a feature">
                                    <button type="button" class="btn btn-success add-feature-btn-edit">
                                        <i class="mdi mdi-plus"></i>
                                    </button>
                                </div>
                            </div>
                            <small class="text-muted">Click + to add more features</small>
                        </div>

                        <div class="form-group">
                            <div class="form-check form-switch">
                                <input type="checkbox" name="is_popular" class="form-check-input" id="edit_is_popular"
                                    role="switch" <?php echo $edit_plan['is_popular'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="edit_is_popular">
                                    Mark as Popular Plan
                                </label>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <a href="plans.php" class="btn btn-light">Cancel</a>
                        <button type="submit" name="edit_plan" class="btn btn-primary text-white">
                            <i class="mdi mdi-content-save"></i> Update Plan
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- SweetAlert CDN -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
    // Feature Management
    document.addEventListener('DOMContentLoaded', function () {
        // Add feature field for add modal
        const addFeatureBtn = document.querySelector('.add-feature-btn');
        if (addFeatureBtn) {
            addFeatureBtn.addEventListener('click', function () {
                addFeatureField('add-features-container');
            });
        }

        // Add feature field for edit modal
        const addFeatureBtnEdit = document.querySelector('.add-feature-btn-edit');
        if (addFeatureBtnEdit) {
            addFeatureBtnEdit.addEventListener('click', function () {
                addFeatureField('edit-features-container');
            });
        }

        // Remove feature field
        document.addEventListener('click', function (e) {
            if (e.target.classList.contains('remove-feature-btn')) {
                e.target.closest('.input-group').remove();
            }
        });

        // Auto-show edit modal if edit_id is set
        <?php if ($edit_plan): ?>
            const editModal = new bootstrap.Modal(document.getElementById('editPlanModal'));
            editModal.show();
        <?php endif; ?>
    });

    function addFeatureField(containerId, value = '') {
        const container = document.getElementById(containerId);
        const newInput = document.createElement('div');
        newInput.className = 'input-group mb-2';
        newInput.innerHTML = `
        <input type="text" name="features[]" class="form-control" value="${value}" placeholder="Add a feature">
        <button type="button" class="btn btn-danger remove-feature-btn">
            <i class="mdi mdi-minus"></i>
        </button>
    `;
        container.appendChild(newInput);
    }

    // Delete Plan with SweetAlert
    function deletePlan(planId, planName) {
        Swal.fire({
            title: 'Are you sure?',
            text: `You are about to delete "${planName}". This action cannot be undone!`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#06417c',
            confirmButtonText: 'Yes, delete it!',
            cancelButtonText: 'Cancel',
            background: '#fff',
            backdrop: 'rgba(0,0,0,0.4)'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = `plans.php?action=delete&id=${planId}`;
            }
        });
    }

    // Toggle Plan Status with SweetAlert
    function togglePlan(planId, currentStatus) {
        const action = currentStatus === 'active' ? 'deactivate' : 'activate';
        const actionText = currentStatus === 'active' ? 'This plan will no longer be available for purchase.' : 'This plan will become available for purchase.';

        Swal.fire({
            title: `${action.charAt(0).toUpperCase() + action.slice(1)} Plan?`,
            text: actionText,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#06417c',
            cancelButtonColor: '#d33',
            confirmButtonText: `Yes, ${action} it!`,
            cancelButtonText: 'Cancel',
            background: '#fff',
            backdrop: 'rgba(0,0,0,0.4)'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = `plans.php?action=toggle&id=${planId}`;
            }
        });
    }
</script>

<style>
    .pricing-card {
        position: relative;
        transition: all 0.3s ease;
        border: none;
        border-radius: 15px;
        box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
        margin-bottom: 30px;
        overflow: hidden;
    }

    .pricing-card:hover {
        transform: translateY(-10px);
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.15);
    }

    .pricing-card.popular {
        border: 3px solid #3085d6;
        transform: scale(1.02);
    }

    .pricing-card.popular:hover {
        transform: scale(1.02) translateY(-10px);
    }

    .popular-badge {
        position: absolute;
        top: 20px;
        right: -35px;
        background: #3085d6;
        color: white;
        padding: 8px 40px;
        font-size: 12px;
        font-weight: bold;
        transform: rotate(45deg);
        box-shadow: 0 2px 10px rgba(76, 175, 80, 0.3);
        z-index: 1;
    }

    .card-header {
        border-radius: 15px 15px 0 0 !important;
        border: none;
        padding: 30px 20px;
    }

    .card-header h4 {
        font-size: 1.8rem;
        font-weight: 700;
    }

    .price {
        font-size: 3rem;
        font-weight: 800;
        margin-bottom: 0.5rem;
    }

    .price small {
        font-size: 1.2rem;
        font-weight: 400;
    }

    .members-count {
        margin: 20px 0;
    }

    .features-list {
        max-height: 400px;
        overflow-y: auto;
        padding: 10px 0;
    }

    .feature-text {
        font-size: 0.95rem;
        color: #555;
    }

    .action-buttons .btn-group {
        border-radius: 10px;
        overflow: hidden;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    }

    .action-buttons .btn {
        border: none;
        padding: 12px 0;
        font-weight: 600;
        transition: all 0.3s ease;
    }

    .action-buttons .btn:hover {
        transform: translateY(-2px);
    }

    /* Plan Specific Styles */
    .basic-plan .card-header {
         background: linear-gradient(135deg, #0034c4 0%, #0033c4 100%) !important;
    }

    .standard-plan .card-header {
         background: linear-gradient(135deg, #0034c4 0%, #0033c4 100%) !important;
    }

    .premium-plan .card-header {
          background: linear-gradient(135deg, #0034c4 0%, #0033c4 100%) !important;
    }

    .custom-plan .card-header {
        background: linear-gradient(135deg, #0034c4 0%, #0033c4 100%) !important;
    }

    /* Scrollbar Styling */
    .features-list::-webkit-scrollbar {
        width: 6px;
    }

    .features-list::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 10px;
    }

    .features-list::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 10px;
    }

    .features-list::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }

    /* Modal Enhancements */
    .modal-header {
        border-radius: 15px 15px 0 0;
    }

    .modal-content {
        border-radius: 15px;
        border: none;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
    }

    .input-group .btn {
        border-radius: 0 8px 8px 0;
    }

    .form-control {
        border-radius: 8px;
        padding: 12px 15px;
        border: 2px solid #e9ecef;
        transition: all 0.3s ease;
    }

    .form-control:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
    }

    .btn-primary {
        background-color: #0033C4 !important;
        border-color: #0033C4 !important;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .pricing-card {
            margin-bottom: 20px;
        }

        .price {
            font-size: 2.5rem;
        }

        .action-buttons .btn {
            padding: 10px 0;
            font-size: 0.9rem;
        }
    }
</style>